// this file contains the device specific equates for the Vaunix LSG synthesizer
//
// Derived from RSD 12-2006 initial version from Windows API
// HME 02-20-2010 Copied from DLL project code for the Linux user version
// RSD 4-15-2013 updated for LSG synthesizers
// !untested!
//-----------------------------------------------------------------------------

// Format of the status report, for the convenience of host side coders

typedef struct 
{
  char pkt_status;
  char count;
  long frequency;
  char dev_status;
  signed char power;
} VNX_STATUS_REPORT;

#define STATUS_PLL_LOCK 0x80 	// MASK: Pll lock status bit, 1 = locked
#define STATUS_NEW_PARAM 0x40 	// MASK: A parameter was set since the last "Save Settings" command
#define STATUS_CMD_OK 0x20 		// MASK: The command was completed by the LSG device without problems
								// !!! Note !!! renamed but value unchanged to avoid a name collision
								// with the status return code for the DLL itself.
#define STATUS_RF_ON 0x10 		// MASK: The RF HW is on (moved from 0x08)


// Bit masks and equates for the Sweep command byte (stored in Sweep_mode, and reported also in Status)

#define SWP_DIRECTION 0x04		// MASK: bit = 0 for sweep up, 1 for sweep down 
#define SWP_CONTINUOUS 0x02		// MASK: bit = 1 for continuous sweeping
#define SWP_ONCE 0x01			// MASK: bit = 1 for single sweep

// HID report equates
#define HR_BLOCKSIZE 6 			// size of the block of bytes buffer in our HID report
#define HID_REPORT_LENGTH 8 	// use an 8 byte report..


typedef struct 
{
  char reportid;				// an artifact of the Windows HID API
  char status;
  char count;
  char byteblock[HR_BLOCKSIZE];
} HID_REPORT1;



typedef struct 
{
  char reportid;				// an artifact of the Windows HID API
  char command;
  char count;
  char byteblock[HR_BLOCKSIZE];
} HID_REPORT_OUT;


// Misc. constants used for the Vaunix devices
#define VNX_MIN_DWELLTIME 10	// 10 ms. minimum dwell time
#define MAX_ATTEN 63 			// we've got 63 db max attenuation -- we represented it as 4 * the value
								// or the attentuation in .25db steps

// Misc commands to send to the device
// For the input reports the first byte is the status, for the output it is the command. The high bit sets the 
// direction.
//
//count is the number of valid bytes in the byteblock array
// byteblock is an array of bytes which make up the value of the command's argument or arguments.
//
// For result reports the command portion of the status byte is equal to the command sent if the command was successful.
// status byte layout:

// Bit0 - Bit5 = command result, equal to command if everything went well
// Bit6 = --reserved--
// Bit7 = --reserved--

// All frequency related data items are DWORD (unsigned) quantities, stored in normal Microsoft byte order.
// Sweep time is a DWORD (unsigned)



// Misc commands to send to the device

#define VNX_SET 0x80
#define VNX_GET 0x00 		// the set and get bits are or'd into the msb of the command byte

// ---- LSG Commands, some used in other devices ----
#define VNX_FREQUENCY 0x04 	// output frequency in 100Khz units
#define VNX_FDWELL 0x05 	// time to dwell at each frequency in the sweep, in 1ms units, minimum of 10ms.
#define VNX_FSTART 0x06 	// frequency lower limit of sweep in 100Khz units 
#define VNX_FSTOP 0x07 		// frequency upper limit of sweep in 100Khz units
#define VNX_FSTEP 0x08 		// frequency step size, in 100Khz units
#define VNX_SWEEP 0x09 		// command to start/stop sweep, data = 01 for single sweep, 00 to stop
							// sweeping, and 02 for continuous sweeping.
							// For the DSS synth, there are more sweep option bits

#define VNX_RFMUTE 0x0A 	// enable or disable RF output, byte = 01 to enable, 00 to disable
#define VNX_ATTEN 0x0B 		// attenuator setting, byte = number of dbm attenuation in .25 db steps
							// 00 is no attenuation, 02 is .5 db, 04 is 1 db

#define VNX_SAVEPAR 0x0C 	// command to save user parameters to flash, data bytes must be
							// set to  0x42, 0x55, 0x31 as a key to enable the flash update
							// all of the above settings are saved (Frequency, sweep params, etc.

#define VNX_PWR 0x0D 		// power output setting, relative to calibrated value - adds to calibrated
							// attenuator setting. It is a byte, with the attenuation expressed in .25db
							// steps. Thus, 2 = .5 db attenuation, 4 = 1.0db attenuation, etc. 

#define VNX_DEFAULTS 0x0F 	// restore all settings to factory default
							// FSTART = Minimum Frequency, FSTOP = Maximum Frequency

#define VNX_STATUS 0x0E 	// Not really a command, but the status byte value for periodic status reports.

#define VNX_INTOSC 0x23 	// enable/disable internal reference oscillator - Byte == 01 for on, 00 for off


// ---------------------- Misc. commands ---------------------------


#define VNX_CALPWR 0x1B 		// set/get the calibrated power output, value is in the first data byte
								// this is saved by the SAVECAL command
#define VNX_MAX_PWR 0x1B 		// just an alias to VNX_CALPWR

#define VNX_MINFREQUENCY 0x20 	// get (no set allowed) the minimum frequency
#define VNX_MAXFREQUENCY 0x21 	// get (no set allowed) the maximum frequency

#define VNX_MODELNAME 0x22 		// get (no set allowed) the device's model name string -- last 6 chars only



// ----------- Global Equates ------------
#define MAXDEVICES 64
#define MAX_MODELNAME 32

// ----------- Data Types ----------------

#define DEVID unsigned int

typedef struct
{
  int DevStatus;
  int Frequency;
  int MinFrequency;
  int MaxFrequency;
  int StartFrequency;
  int EndFrequency;
  int FrequencyStep;
  int DwellTime;
  int Power;
  int MaxPower;
  int MinPower;
  int Modebits;
  int SerialNumber;
  char ModelName[MAX_MODELNAME];
  /* so we can find this device again later */
  unsigned int idVendor;
  unsigned int idProduct;
  char Serialstr[16];
  char thread_command;
  char outbuff[8];
  char pending;
  int MyDevID;

} LSGPARAMS;

// ----------- Mode Bit Masks ------------

#define MODE_RFON 0x00000010 		// bit is 1 for RF on, 0 if RF is off
#define MODE_INTREF 0x00000020 		// bit is 1 for internal osc., 0 for external reference
#define MODE_SWEEP 0x0000000F 		// bottom 4 bits are used to keep the sweep control bits


// ----------- Command Equates -----------


// Status returns for commands
#define LVSTATUS unsigned int

#define STATUS_OK 0
#define BAD_PARAMETER  0x80010000 		// out of range input -- frequency outside min/max etc.
#define BAD_HID_IO     0x80020000
#define DEVICE_NOT_READY  0x80030000 	// device isn't open, no handle, etc.

// Status returns for DevStatus

#define INVALID_DEVID 0x80000000 		// MSB is set if the device ID is invalid
#define DEV_CONNECTED 0x00000001 		// LSB is set if a device is connected
#define DEV_OPENED 0x00000002 			// set if the device is opened
#define SWP_ACTIVE 0x00000004 			// set if the device is sweeping
#define SWP_UP 0x00000008 				// set if the device is sweeping up in frequency
#define SWP_REPEAT 0x00000010 			// set if the device is in continuous sweep mode

// Internal values in DevStatus
#define DEV_LOCKED   0x00002000 		// set if we don't want read thread updates of the device parameters
#define DEV_RDTHREAD   0x00004000 		// set when the read thread is running


void fnLSG_Init(void);
void fnLSG_SetTestMode(bool testmode);
int fnLSG_GetNumDevices();
int fnLSG_GetDevInfo(DEVID *ActiveDevices);
int fnLSG_GetModelName(DEVID deviceID, char *ModelName);
int fnLSG_InitDevice(DEVID deviceID);
int fnLSG_CloseDevice(DEVID deviceID);
int fnLSG_GetSerialNumber(DEVID deviceID);


LVSTATUS fnLSG_SetFrequency(DEVID deviceID, int frequency);

LVSTATUS fnLSG_SetStartFrequency(DEVID deviceID, int startfrequency);
LVSTATUS fnLSG_SetEndFrequency(DEVID deviceID, int endfrequency);
LVSTATUS fnLSG_SetFrequencyStep(DEVID deviceID, int frequencystep);
LVSTATUS fnLSG_SetDwellTime(DEVID deviceID, int dwelltime);
LVSTATUS fnLSG_SetPowerLevel(DEVID deviceID, int powerlevel);
LVSTATUS fnLSG_SetRFOn(DEVID deviceID, bool on);

LVSTATUS fnLSG_SetUseInternalRef(DEVID deviceID, bool internal);
LVSTATUS fnLSG_SetSweepDirection(DEVID deviceID, bool up);
LVSTATUS fnLSG_SetSweepMode(DEVID deviceID, bool mode);
LVSTATUS fnLSG_StartSweep(DEVID deviceID, bool go);
LVSTATUS fnLSG_SaveSettings(DEVID deviceID);

int fnLSG_GetFrequency(DEVID deviceID);
int fnLSG_GetStartFrequency(DEVID deviceID);
int fnLSG_GetEndFrequency(DEVID deviceID);
int fnLSG_GetDwellTime(DEVID deviceID);
int fnLSG_GetFrequencyStep(DEVID deviceID);

int fnLSG_GetRF_On(DEVID deviceID);
int fnLSG_GetUseInternalRef(DEVID deviceID);
int fnLSG_GetPowerLevel(DEVID deviceID);
int fnLSG_GetMaxPwr(DEVID deviceID);
int fnLSG_GetMinPwr(DEVID deviceID);
int fnLSG_GetMaxFreq(DEVID deviceID);
int fnLSG_GetMinFreq(DEVID deviceID);


char* fnLSG_perror(LVSTATUS status);
char* fnLSG_LibVersion(void);
